#!/usr/bin/env python3
import pandas as pd, numpy as np, re

IN_CSV = "outputs/lensing_plateau.csv"

def rg_mid(s):
    s=(s or "").strip().replace("—","-").replace("–","-")
    m=re.match(r"\s*([0-9.]+)\s*-\s*([0-9.]+)\s*", s)
    if not m: return np.nan
    a,b=float(m.group(1)), float(m.group(2))
    return 0.5*(a+b)

def wls_slope(x,y,w):
    x=np.asarray(x,float); y=np.asarray(y,float); w=np.asarray(w,float)
    W=w.sum()
    if not np.isfinite(W) or W<=0: return np.nan
    xb=(w*x).sum()/W; yb=(w*y).sum()/W
    num=(w*(x-xb)*(y-yb)).sum(); den=(w*(x-xb)**2).sum()
    return num/den if den>0 else np.nan

def main():
    df=pd.read_csv(IN_CSV)
    df=df[df["claimable"].astype(str).str.lower()=="true"].copy()
    if df.empty: print("no claimables"); return
    df["RG_mid"]=df["R_G_bin"].apply(rg_mid)
    df["A_theta"]=pd.to_numeric(df["A_theta"], errors="coerce")
    df["rmse_flat"]=pd.to_numeric(df["rmse_flat"], errors="coerce")
    df=df.replace([np.inf,-np.inf],np.nan).dropna(subset=["RG_mid","A_theta","rmse_flat"])
    if df.empty: print("no usable rows"); return

    for ms, g in df.groupby("Mstar_bin"):
        x=g["RG_mid"].to_numpy(); y=g["A_theta"].to_numpy()
        w=1.0/(g["rmse_flat"].to_numpy()**2 + 1e-8)
        s_full=wls_slope(x,y,w)
        print(f"\nMstar_bin {ms}: n={len(g)}  full_slope={s_full:+.4e}")
        if len(g)<3: 
            print("  (need >=3 for sensitivity)"); 
            continue
        # leave-one-out
        rows=[]
        for i,(_,r) in enumerate(g.iterrows()):
            m=wls_slope(np.delete(x,i), np.delete(y,i), np.delete(w,i))
            rows.append((r["stack_id"], r["R_G_bin"], r["A_theta"], r["rmse_flat"], m, s_full-m))
        rows.sort(key=lambda z: abs(z[-1]), reverse=True)
        print("  influence (largest first):")
        print("    stack_id | R_G_bin | A_theta | rmse | slope_wo_i | delta")
        for sid,rg,a,rm,sw,delta in rows:
            print(f"    {sid} | {rg} | {a:+.3e} | {rm:.3e} | {sw:+.4e} | {delta:+.4e}")

if __name__=="__main__":
    main()
